<?php

namespace YOOtheme\Demo\Routes;

use Joomla\CMS\Categories\Categories;
use Joomla\CMS\Helper\TagsHelper;
use ReflectionMethod;
use YOOtheme\Builder;
use YOOtheme\Builder\Joomla\Source\Listener\LoadBuilderConfig;
use YOOtheme\Builder\Source;
use YOOtheme\Config;
use YOOtheme\Event;
use YOOtheme\GraphQL\Type\Definition\ListOfType;
use YOOtheme\GraphQL\Type\Definition\ScalarType;
use YOOtheme\Http\Request;
use YOOtheme\Metadata;
use YOOtheme\Storage;
use function YOOtheme\app;

class ValidateSource
{
    public static function match(Request $request)
    {
        return $request->getQueryParam('validateSource');
    }

    public static function handle($demo)
    {
        Event::on('theme.init', function () {
            app(Config::class)->set('app.debug', true);
        });

        $i = 0;
        $log = function ($errors) use (&$i) {
            return app(Metadata::class)->set("script:graphql-errors{$i}", json_encode($errors), [
                'type' => 'application/json',
                'data-source-errors' => $i++,
            ]);
        };

        Event::on('source.error', $log);

        app()->extend(Builder::class, function (Builder $builder) use ($log) {
            $templates = app(Storage::class)('templates', []);

            $builder->addTransform('preload', function ($node, array &$params) use (
                $log,
                $builder,
                $templates,
            ) {
                $logFn = $log;
                $log = fn($error) => $logFn([['message' => "{$error} ({$params['prefix']})"]]);

                $matchCategories = function ($categories, $params) use ($log) {
                    $parents = [];

                    foreach (is_array($categories) ? $categories : [$categories] as $catid) {
                        $category = Categories::getInstance('content', [
                            'countItems' => false,
                        ])->get($catid);

                        $path = implode(
                            ' - ',
                            array_reverse(
                                array_map(function ($node) {
                                    if (isset($node->name)) {
                                        return $node->type . ' ' . $node->name;
                                    }
                                    return $node->type;
                                }, $params['path']),
                            ),
                        );
                        $path = "`{$params['prefix']}` in `{$path}`";

                        if (!$category) {
                            $log("{$catid} is not a valid category id.");
                            continue;
                        }

                        if (
                            !in_array($category->parent_id, $parents) &&
                            $category->parent_id !== 'root'
                        ) {
                            $parent = $category->getParent();

                            if (!$parent) {
                                continue;
                            }

                            $children = array_map(function ($cat) {
                                return (string) $cat->id;
                            }, $parent->getChildren());

                            $parents[] = $parent->id;
                        }

                        // Warn if all child categories are selected
                        if (
                            empty($category->getChildren()) &&
                            !empty($children) &&
                            !array_diff($children, (array) $categories)
                        ) {
                            $log(
                                "All child categories of `{$parent->id}: {$parent->title}` are selected on {$path}.",
                            );
                        }
                    }
                };

                $matchTags = function ($tags) use ($log) {
                    foreach (is_array($tags) ? $tags : [$tags] as $tag) {
                        if (!(new TagsHelper())->getTagNames([$tag])) {
                            $log("{$tag} is not a valid tag id.");
                        }
                    }
                };

                if (
                    $node->type === 'layout' &&
                    str_starts_with($params['prefix'] ?? '', 'template-')
                ) {
                    $template = $templates[substr($params['prefix'], 9)];
                    $matchCategories($template['query']['catid'] ?? [], $params);
                    $matchTags($template['query']['tag'] ?? []);

                    // now reflection for protected call() method
                    $reflectedMethod = new ReflectionMethod(
                        LoadBuilderConfig::class,
                        'getTemplates',
                    );
                    $templates = $reflectedMethod->invoke(null);

                    // Look for obsolete query params
                    foreach ($template['query'] ?? [] as $key => $value) {
                        if (
                            !isset(
                                $templates[$template['type']]['fieldset']['default']['fields'][
                                    $key
                                ],
                            )
                        ) {
                            $log("{$key} is not a valid template query param.");
                        }
                    }
                }

                $matchCategories($node->source->query->arguments->catid ?? [], $params);
                $matchTags($node->source->query->arguments->tags ?? []);

                if (isset($node->source->query->name, $node->source->query->arguments)) {
                    try {
                        $field = null;
                        $query = app(Source::class)->getSchema()->getConfig()->getQuery();
                        foreach (explode('.', $node->source->query->name) as $name) {
                            $field = ($field ? $field->getType() : $query)->getField($name);
                        }
                    } catch (\Exception $e) {
                        $log($e->getMessage());
                        return;
                    }

                    foreach ($node->source->query->arguments as $name => $value) {
                        $argument = $field->getArg($name);
                        if (!$argument) {
                            $log(
                                "{$name} is not a valid argument for {$node->source->query->name}.",
                            );
                            continue;
                        }

                        $type = $argument->getType();

                        if ($type instanceof ListOfType) {
                            if (!is_array($value)) {
                                $log(
                                    "Argument {$name} has invalid value for {$node->source->query->name}.",
                                );
                            }
                        }

                        if ($type instanceof ScalarType) {
                            if (!is_scalar($value)) {
                                $log(
                                    "Argument {$name} has invalid value for {$node->source->query->name}.",
                                );
                            }
                        }
                    }
                }

                // Look for obsolete props
                $type = $params['type'];
                $fields = $type->fields ?? [];
                foreach ($type->panels ?? [] as $panel) {
                    if (isset($panel['fields'])) {
                        $fields += $panel['fields'];
                    }
                }

                // allowed props
                $types = [
                    'column' => [
                        'order_first' => null,
                        'width_default' => null,
                        'width_small' => null,
                        'width_medium' => null,
                        'width_large' => null,
                        'width_xlarge' => null,
                        'media_overlay_gradient' => null,
                    ],
                    'section' => [
                        'media_overlay_gradient' => null,
                    ],
                    'newsletter' => [
                        'provider' => null,
                    ],
                ];
                if (isset($types[$node->type])) {
                    $fields += $types[$node->type];
                }

                foreach ($node->props as $name => $value) {
                    if (!array_key_exists($name, $fields) && !str_contains($name, 'parallax_')) {
                        $log("{$name} is not a valid prop in {$node->type}.");
                    }
                }
            });
        });
    }
}
